#-----------------------------------------------------------
# Lecture 17 -- Least Square Fitting Examples
#-----------------------------------------------------------

import numpy as np
import matplotlib.pyplot as plt

#-----------------------------------------------------------
# A. Fitting
#-----------------------------------------------------------

#%% (i) Polynomial fits
#
# Steps to fitting a polynomial:
# (1) Set or read in your given x and y data that you want to fit
# (2) Use polyfit to get the coefficients
#       p = np.polyfit(x, y, deg)
#       p: numpy array of coefficients
#       x: x data
#       y: y data
#       deg: polynomial order (e.g. linear == 1, quadratic == 2, etc.)
# (3) Evaluate the polynomial using either 
#       (1) a function defining p(x) or 
#       (2) np.polyval(p, x)
#
# Note: The polynomial is given by
#   p(x) = p[0]*x**deg + p[1]*x**(deg-1) + ... p[deg]
#   ==> the coefficients in the `p` array go from largest to smallest power
# 
# Additional information about polyfit and polyval at the links:
#   (polyfit) https://docs.scipy.org/doc/numpy/reference/generated/numpy.polyfit.html 
#   (polyval) https://docs.scipy.org/doc/numpy/reference/generated/numpy.polyval.html#numpy.polyval
#
# ** Example **
# Fit the following x and y data using a 3rd order polynomial.

# (1) set the x & y data (this is normally given to you or measured from experiment)
x_given = np.array([0., 1., 2., 3., 4., 5.])         # given x data
y_given = np.array([0, 0.8, 0.9, 0.1, -0.8, -1.0])   # given y data

# (2) get the coefficients
p3  = np.polyfit(x_given,y_given, 3)   # a polynomial object
print("3rd order polyominal fit coefficients (highest power first) =", p3)

# (3) evaluate the polynomial using polyfit
x_p = np.linspace(-0.5,6.0, 100)       # x data for plotting the poly fit
y_p = np.polyval(p3, x_p)              # call polyval

# make a plot
plt.figure()
plt.plot(x_given, y_given,   'o')
plt.plot(x_p,     y_p,       '-')
plt.legend(['data', 'polyfit'], loc='best')
plt.xlabel('x'); plt.ylabel('y')
plt.ylim(-2,2);
plt.show()

#%% (ii) Coefficient of Determination, R^2
#
# There isn't really a convenient R2 function in numpy. But it is easy enough 
# to define one for ourselves.
#
#   Remember: R2 = 1 - SSE/SST
#   SSE = sum_i (y_i - f(x_i))^2
#   SST = sum_i (y_i - avg(y_i))^2

def rsq(y_data, y_pred):
    """
    rsq -- calculates the coefficient of determination (R^2) for a fit
           
    y_data: y data, i.e. y_i
    y_pred: predicted values from the model, i.e. f(x_i)
    """
    
    SSE = np.sum( (y_data - y_pred)**2 )
    SST = np.sum( (y_data - np.average(y_data))**2 )
    return 1 - SSE/SST

# example using rsq from polynomial fit above
y_pred = np.polyval(p3, x_given) # evaluate only at the x_i data points
print('R2 = ', rsq(y_given, y_pred))

#%%
#-----------------------------------------------------------
# Practice
#-----------------------------------------------------------

# Load the given x and y data from the file "Lec17-data.dat" 
# and fit it to a fourth order polynomial. Make a plot of the data and the fit.
# Evaluate R2 and comment on how well the polynomial fits the data?


#%% (iii) General Curve Fits
#
# * We can fit a general function f(x; a, b, c) where f is a function of x 
#   with parameters a, b, c that we want to optimize for a set of given data.
# 
# * Use `curve_fit` available from `from scipy.optimize import curve_fit`
# 
# Steps to fitting a general curve using curve_fit:
# (1) import module: `from scipy.optimize import curve_fit`
# (2) Set or read in your given x and y data that you want to fit
# (3) Define the function with the parameters
#       def <function name>( x, param1, param2, ... ):
#            return <put function here>
# (4) Use `curve_fit` to fit the data
#       params, stats = curve_fit( <function name>, x_data, y_data )
#       params: holds the parameters after the fit (param1, param2, ...)
#       stats: you can ignore these values (covariance matrix)
#       <function name>: the function from the last step
#       x_data, y_data: given data from the first step
# (5) Evaluate the function using the params from (4) and the function 
#     from (3).
# 
# Additional information about curve_fit at the link:
#   (curve_fit) https://docs.scipy.org/doc/scipy/reference/generated/scipy.optimize.curve_fit.html
#
# ** Example **
# Fit the function f(x) = a exp(-bx) + c to the following data

# (1) import module
import scipy.optimize as opt

# (3) Define the function with parameters
# ** Out of order because I am using this to define the given data.
def f(x, a, b, c) :     
    return a*np.exp(-b*x) + c

# (2) Set the given data (this is normally given to you or measured from experiment)
x_given = np.linspace(0,4,50)
y_given = f(x_given,2.5,1.3,0.5) + 0.2*np.random.normal(size=len(x_given))

# (4) Do the curve fit
params, stats = opt.curve_fit(f, x_given, y_given)

# (5) Output / plot the results

y_fit =  f(x_given,params[0],params[1],params[2])

plt.figure()
plt.plot(x_given, y_given, 'o')
plt.plot(x_given, y_fit, 'k-')
plt.legend(['data', 'fit'], loc='best')
plt.xlabel('x')
plt.ylabel('y, f(x)')
plt.text(0.6, 2.7, "a=%.4f, b=%.4g, c=%.4f"%(params[0], params[1], params[2]), fontsize=12)
plt.show()

