#-----------------------------------------------------------
# Lecture 18 -- Example -- Interpolation Procedure in Python
#-----------------------------------------------------------

import numpy as np
import matplotlib.pyplot as plt

#%%
#-----------------------------------------------------------
# A. Interpolation
#-----------------------------------------------------------
#
# An interpolation is a way to get information in between discrete data points.
# For instance, linear interpolation amounts to drawing straight lines between
# points and using those lines to predict the value. A *spline* is a term for 
# a piecewise, polynomial interpolation function that *goes through all of 
# the data*.
#
# Python has a 1D interpolation function in the module scipy.interpolate
# called: `interp1d`
#
# Steps to do an spline interpolation:
# (1) Import the module: `from scipy.interpolate import interp1d`
# (2) Read in or define the given x and y data (e.g. from experiment)
# (3) Call interp1d to create an *interpolation function*
#       <interp fn> = interp1d( x_data, y_data, kind = 'name of order')
#       <interp fn>: function you can use to get interpolated data
#       x_data, y_data: given data
#       kind: order of the spline, e.g. ('linear', 'quadratic', 'cubic')
# (4) Evaluate function at any x_i between the bounds of the data using:
#     `y_i = <interp fn>(x_i)`
#
# Additional information about interp1d at the link:
#   (interp1d) https://docs.scipy.org/doc/scipy-0.19.1/reference/generated/scipy.interpolate.interp1d.html
#
# ** Example 1 **
# Interpolate the following x and y data to find the y value at x = 1.5

# (1) import module
from scipy.interpolate import interp1d

# (2) Define x and y data
x_given = np.array([0,1,2,3,4,5,6,7,8,9,10])
y_given = np.cos(x_given**2.0/8.0)

# (3) Call interp1d to get interpolation function
f_linear = interp1d(x_given, y_given) # with no kind, it defaults to linear

# (4) Evaluate the function at the desired x value
x_intermediate = 1.5
y_intermediate = f_linear(x_intermediate)

print('y_below = ', y_given[1])
print('y_avove = ', y_given[2])
print('y_intermediate = ', y_intermediate)

plt.figure(figsize=(10,8))
plt.plot(x_given, y_given, 'o',label='given data')
plt.plot(x_given[1:3], y_given[1:3], '-',label='')
plt.plot(x_intermediate, y_intermediate,'x', label='interpolated value')
plt.ylabel('x')
plt.ylabel('y')
plt.legend(loc='best', frameon=False);
plt.show()

#%%
# ** Example 2**
# 
# This example illustrates the difference between different spline orders 
# (linear versus cubic)
#
# * Again, we have a discrete set of given data points: x and y
# * Plot the points and the underlying function.

def f_exact(x):
    return np.cos(x**2/8.0)

# Given data
N_given = 8 # How many data points we are given. 
            # The spline does better with more points.
            
x_given = np.linspace(0, 10, N_given) # Interpolate between the given data
y_given = np.cos(x_given**2.0/8.0)    

# Linear interpolation function
f_linear   = interp1d(x_given, y_given) # get an interpolation function

# Spline interpolation function
f_spline   = interp1d(x_given, y_given, kind='cubic')

# Plot the results
xx = np.linspace(0,10,200) # interpolate *many* points to show the functions

plt.plot(x_given,y_given,'o')
plt.plot(xx, f_linear(xx), '-')
plt.plot(xx, f_spline(xx), '--')
plt.plot(xx, f_exact(xx), ':')
plt.legend(['data', 'linear', 'spline', \
            'exact function'], loc='best', frameon=False);

# %%
#-----------------------------------------------------------
# Practice
#-----------------------------------------------------------

# Load the data in the file 'Lec_18-data.csv'. Use a linear 
# interpolation and cubic spline to get the value for y at x = 2.5.
    
# (Scroll down for key)
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
#%%
# Answer key for Practice Problem
x_data, y_data = np.loadtxt('Lec18-data.csv', delimiter=',', unpack=True)

f_linear = interp1d(x_data, y_data, kind='linear')
f_cubic = interp1d(x_data, y_data, kind='cubic')

x = 2.5
print('x = ', x)
print('y (linear) = ', f_linear(x)) 
print('y (cubic) = ', f_cubic(x)) 

