#-----------------------------------------------------------
# Lecture 18 -- Interpolation Concepts
#-----------------------------------------------------------

import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit
from scipy.interpolate import interp1d

# These examples demonstrates when fitting is appropriate and when 
# interpolation is appropriate. Don't worry if you don't know how
# to do fitting or interpolation yet. That is the next section.

# Only leave one of these functions un-commented:

# **Does not** show Runge phenomenon
#def f(x, a, b, c):
#    return np.exp(-x/a)*np.cos(np.pi*x/b) + c

# **Does** show Runge phenomenon
def f(x, a, b, c):
    return np.cos(np.pi*x/b)*(np.cosh(np.pi*x/a))**(-2) + c 

A = 5
B = 2
C = 3

x_lo = -5
x_hi = 5

#%%
#-----------------------------------------------------------
# A. Fitting
#-----------------------------------------------------------
# 
# * We use fitting when the data is noisy and we want to fit a single function
#   that approximately captures all of the data.
# * To do a fit, we must have **more data points than parameters**

np.random.seed(1234)    # set a random seed so we get the 
                        # same "random" values every time

# make up "data"
x_i = np.linspace(x_lo, x_hi, 50)
e_i = 0.25*(np.random.rand(len(x_i))-0.5)
y_i = f(x_i, A, B, C) + e_i

# fit to the data
params, stats = curve_fit(f, x_i, y_i, p0 = [A, B, C])
print('A_fit = ', params[0])
print('B_fit = ', params[1])
print('C_fit = ', params[2])

plt.figure(figsize=(10.,8.))
plt.plot(x_i, y_i, 'bo')
plt.plot(x_i, f(x_i, A, B, C), 'k--')
plt.plot(x_i, f(x_i, params[0], params[1], params[2]), 'r-')
plt.title('Least-Squares Fitting')
plt.show()

#%%
#-----------------------------------------------------------
# B. Interpolation
#-----------------------------------------------------------
# 
# (i) Polynomial Uniqueness Thoerem
#
# * Let's start with a mathematical fact (the Polynomial Uniqueness Theorem):
#
#       A polynomial of order `n` is uniquely determined by `n+1` points.
#
# * For example, two points determine a line, three points determine a quadratic,
#   four points determines a cubic and so on.
#
# * Why is this? A quick and dirty argument (not a rigorous proof):
#
#   (linear case)
#   y = a x + b ==> 2 unknowns (a, b)
#
#   2 points gives us two equations:
#   y1 = a x1 + b, 
#   y2 = a x2 + b 
#   ==> can solve for a, b
#
#   (quadratic case)
#   y = a x^2 + b x + c ==> 3 unknowns (a, b, c); 
#
#   3 points gives us three equations:
#   y1 = a x1^2 + b x1 + c
#   y2 = a x2^2 + b x2 + c
#   y3 = a x3^2 + b x3 + c
#   ==> can solve for a, b, c
#
#   etc.
#

# (ii) Runge's Phenomenon
#
# * Why not use a single polynomial to exactly fit the data?
# * Because we can get a terrible fit (not always, but sometimes)! 
#   The blow-up at the ends of the function is called Runge's phenomenon.

n_order = 8 # change this to see different order

x_i = np.linspace(x_lo, x_hi, n_order+1)
y_i = f(x_i, 5, 2, 3)

coeffs = np.polyfit(x_i, y_i, n_order)

xx = np.linspace(x_lo , x_hi, 100)
yy = np.polyval(coeffs, xx)

plt.figure()
plt.plot(x_i, y_i, 'o')
plt.plot(xx, f(xx, 5, 2, 3), 'k--')
plt.plot(xx, yy, 'r-')
plt.title('Exact Fit of a Single Polynomial')
plt.show()

#%% (iii) Splines -- Piecewise Polynomial Exact Fits

x_i = np.linspace(x_lo, x_hi, 15)
y_i = f(x_i, A, B, C)

xx = np.linspace(x_lo, x_hi, 100)
yy = f(xx, A, B, C)

f_linear = interp1d(x_i, y_i, kind='linear')
f_quad = interp1d(x_i, y_i, kind='quadratic')
f_cubic = interp1d(x_i, y_i, kind='cubic')

plt.figure(figsize=(10,8))
plt.plot(xx, yy, 'k--') # exact
plt.plot(x_i, y_i, 'bo')
#plt.plot(xx, f_linear(xx), 'r-')
#plt.plot(xx, f_quad(xx), 'g-')
plt.plot(xx, f_cubic(xx), 'b-')
plt.title('Spline -- Exact fit of Piecewise Polynomials')
plt.show()

#%%
#-----------------------------------------------------------
# Practice -- Fitting vs. Interpolation
#-----------------------------------------------------------

# Question 1: 
    
# You are measuring heat capacity data as a function of temperature. Which 
# technique do you use (fitting or interpolation) to get the parameters 
# for the formula: Cp(T) = A*T^3 + B*T^2 + C*T + D ?

# Question 2: 
    
# Say that the heat capacity data you measured in Question 1 is given at 
# regular temperature intervals of 10 K, e.g. 273 K, 283 K, 293 K, etc.
# What technique do you use (fitting or interpolation) to get the heat 
# capacity at 288 K? Does your answer depend on how noisy the data is?

# (Scroll Down for Key)


































# Answer Key:
# Q1:
# You need to use a fitting procedure (e.g. polynomial least squares) 
# to find A, B, C and D.
# Q2:
# It does depend on how noisy the data is. If the data has little noise, 
# an interpolation procedure (e.g. cubic spline) would work very well. 
# However, if the data is very noisy, it would be better to use the fitting
# procedure described above and then evaluate the function at Cp(288 K).

